#include <gtest/gtest.h>

#include <array>
#include <string_view>

#include "Common/BitUtils.h"
#include "Core/IOS/USB/Emulated/Skylanders/SkylanderCrypto.h"
#include "Core/IOS/USB/Emulated/Skylanders/SkylanderFigure.h"

using namespace IOS::HLE::USB::SkylanderCrypto;

// Figure data generated by:
//
//  const std::string temp_dir = File::CreateTempDir();
//  IOS::HLE::USB::SkylanderFigure figure(temp_dir + "/test.sky");
//  figure.Create(0x1D6, 0x3000, std::array<u8, 4>({0x01, 0x23, 0x45, 0x67}));
//
//  IOS::HLE::USB::FigureData data = figure.GetData();
//  data.skylander_data.money = 5000;
//  data.skylander_data.hero_level = 50;
//  data.skylander_data.playtime = 1564894;
//  const std::u16string nickname = UTF8ToUTF16("Test");
//  std::memset(data.skylander_data.nickname.data(), 0, data.skylander_data.nickname.size());
//  std::memcpy(data.skylander_data.nickname.data(), nickname.data(), nickname.size() * 2);
//  data.skylander_data.last_reset.minute = 5;
//  data.skylander_data.last_reset.hour = 7;
//  data.skylander_data.last_reset.day = 11;
//  data.skylander_data.last_reset.month = 3;
//  data.skylander_data.last_reset.year = 2020;
//  data.skylander_data.last_placed.minute = 44;
//  data.skylander_data.last_placed.hour = 8;
//  data.skylander_data.last_placed.day = 14;
//  data.skylander_data.last_placed.month = 4;
//  data.skylander_data.last_placed.year = 2021;
//  figure.SetData(&data);
//
//  data.skylander_data.money = 5600;
//  data.skylander_data.hero_level = 51;
//  data.skylander_data.playtime = 1764894;
//  std::memset(data.skylander_data.nickname.data(), 0, data.skylander_data.nickname.size());
//  std::memcpy(data.skylander_data.nickname.data(), nickname.data(), nickname.size() * 2);
//  data.skylander_data.last_reset.minute = 5;
//  data.skylander_data.last_reset.hour = 7;
//  data.skylander_data.last_reset.day = 11;
//  data.skylander_data.last_reset.month = 3;
//  data.skylander_data.last_reset.year = 2020;
//  data.skylander_data.last_placed.minute = 59;
//  data.skylander_data.last_placed.hour = 9;
//  data.skylander_data.last_placed.day = 14;
//  data.skylander_data.last_placed.month = 4;
//  data.skylander_data.last_placed.year = 2021;
//  figure.SetData(&data);
//
//  std::array<u8, IOS::HLE::USB::FIGURE_SIZE> decrypted = {};
//  figure.DecryptFigure(&decrypted);
//  File::IOFile f(temp_dir + "/decrypted.sky", "wb");
//  f.WriteBytes(decrypted.data(), decrypted.size());
//
static constexpr std::array<u8, IOS::HLE::USB::FIGURE_SIZE> decrypted_figure = {
    0x01, 0x23, 0x45, 0x67, 0x00, 0x81, 0x01, 0x0F, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xD6, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0xCB, 0x7D,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x4B, 0x0B, 0x20, 0x10, 0x7C, 0xCB, 0x0F, 0x0F, 0x0F, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x3C, 0x98, 0xF9, 0x25, 0xA1, 0x7F, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x88, 0x13, 0xDE, 0xE0, 0x17, 0x00, 0x01, 0x88, 0x3C, 0xC4, 0xE3, 0x76, 0xF9,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x54, 0x00, 0x65, 0x00, 0x73, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x1A, 0xF5, 0x2D, 0x76, 0x76, 0xBC, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x2C, 0x08, 0x0E, 0x04, 0xE5, 0x07, 0x00, 0x00, 0x00, 0x00, 0x32, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x05, 0x07, 0x0B, 0x03, 0xE4, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x89, 0xC3, 0xC7, 0xDF, 0x9D, 0x5D, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x54, 0x5B, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xC5, 0x19, 0x6F, 0x78, 0x33, 0xDA, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x56, 0x2F, 0x85, 0xD1, 0xD8, 0x3B, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x70, 0x42, 0x51, 0x82, 0x0F, 0xF8, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xE3, 0x74, 0xBB, 0x2B, 0xE4, 0x19, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x7B, 0xC0, 0xEA, 0x64, 0xB9, 0x16, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0xE0, 0x15, 0x1E, 0xEE, 0x1A, 0x00, 0x02, 0x03, 0xE7, 0xC4, 0xE3, 0x0F, 0x30,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x54, 0x00, 0x65, 0x00, 0x73, 0x00, 0x74, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xE8, 0xF6, 0x00, 0xCD, 0x52, 0xF7, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x3B, 0x09, 0x0E, 0x04, 0xE5, 0x07, 0x00, 0x00, 0x00, 0x00, 0x33, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x05, 0x07, 0x0B, 0x03, 0xE4, 0x07, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xCE, 0x9B, 0xD4, 0x9E, 0x85, 0x34, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x30, 0xD7, 0x02, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x5D, 0xAD, 0x3E, 0x37, 0x6E, 0xD5, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x11, 0x77, 0x96, 0x90, 0xC0, 0x52, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x82, 0x41, 0x7C, 0x39, 0x2B, 0xB3, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0xA4, 0x2C, 0xA8, 0x6A, 0xFC, 0x70, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
    0x37, 0x1A, 0x42, 0xC3, 0x17, 0x91, 0x7F, 0x0F, 0x08, 0x69, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};

// Can be assumed to also mean ComputeCRC48 is correct
TEST(Skylanders, Keygen)
{
  struct
  {
    std::array<u8, 4> nuid;
    u8 sector;
    u64 expected;
  } constexpr inputs[]{{{0x00, 0x00, 0x00, 0x00}, 0, 0x4B0B20107CCB},
                       {{0x94, 0xB0, 0xEE, 0x2D}, 0, 0x4B0B20107CCB},
                       {{0x00, 0x00, 0x00, 0x00}, 11, 0xEA168579FF28},
                       {{0x94, 0xB0, 0xEE, 0x2D}, 1, 0x278e4DA896B5},
                       {{0xF7, 0xDB, 0xFD, 0x5F}, 2, 0x75B9B1F4B9EB}};

  for (auto& test : inputs)
  {
    auto actual = CalculateKeyA(test.sector, test.nuid);
    EXPECT_EQ(test.expected, actual);
  }
}

// Can be assumed to also mean ComputeCRC16 is correct
TEST(Skylanders, Checksums)
{
  std::array<u8, 2> actual = {};
  ComputeChecksumType0(decrypted_figure.data(), actual.data());
  EXPECT_EQ(Common::BitCastPtr<u16>(decrypted_figure.data() + 0x1E),
            Common::BitCastPtr<u16>(actual.data()));

  u16 area_offset = 0x80;

  for (u8 i = 0; i < 2; i++)
  {
    ComputeChecksumType3(decrypted_figure.data() + area_offset + 0x50, actual.data());
    EXPECT_EQ(Common::BitCastPtr<u16>(decrypted_figure.data() + area_offset + 0xA),
              Common::BitCastPtr<u16>(actual.data()));

    ComputeChecksumType2(decrypted_figure.data() + area_offset + 0x10, actual.data());
    EXPECT_EQ(Common::BitCastPtr<u16>(decrypted_figure.data() + area_offset + 0xC),
              Common::BitCastPtr<u16>(actual.data()));

    ComputeChecksumType1(decrypted_figure.data() + area_offset, actual.data());
    EXPECT_EQ(Common::BitCastPtr<u16>(decrypted_figure.data() + area_offset + 0xE),
              Common::BitCastPtr<u16>(actual.data()));

    area_offset += 0x90;

    ComputeChecksumType6(decrypted_figure.data() + area_offset, actual.data());
    EXPECT_EQ(Common::BitCastPtr<u16>(decrypted_figure.data() + area_offset),
              Common::BitCastPtr<u16>(actual.data()));

    area_offset += 0x130;
  }
}

TEST(Skylanders, ToyCode)
{
  const std::array<u8, 11> code_chars = ComputeToyCode(0x14E2CE497CB0B);
  const std::string_view code_string(reinterpret_cast<const char*>(code_chars.data()),
                                     code_chars.size());
  EXPECT_EQ(code_string, "WCJGC-HHR5Q");
}
