/* SPDX-License-Identifier: GPL-2.0-only OR GPL-3.0-only */
/* Copyright (c) 2025 Brett A C Sheffield <bacs@librecast.net> */

#include "test.h"
#include <agent.h>

#define NKEYS 42
#define REDIRECT_BUF f = freopen("/dev/null", "a", stderr); assert(f); setbuf(stderr, buf);
#define REDIRECT_OUT setbuf(stderr, NULL);

static char buf[BUFSIZ];

/* return -1 if found, 0 if not */
static int find_key(lc_keyring_t *keyring, key_combo_t *key)
{
	for (size_t i = 0; i < keyring->nkeys; i++) {
#if 0
		hash_hex_debug(stderr, keyring->key[i], sizeof key->s.pk);
		hash_hex_debug(stderr, key->s.pk, sizeof key->s.pk);
#endif
		if (memcmp(keyring->key[i], key->s.pk, sizeof key->s.pk) == 0) return -1;
	}
	return 0;
}

static int verify_keys(lc_keyring_t *keyring, key_combo_t keys[NKEYS], char *keymap)
{
	for (int i = 0; i < NKEYS; i++) {
		if (!isset(keymap, i)) continue;
		test_assert(find_key(keyring, &keys[i]), "key %i on keyring", i);
	}
	return 0;
}

static void freakeys(lc_keyring_t *keyring)
{
	keyring_freekeys(keyring);
	lc_keyring_free(keyring);
	memset(keyring, 0, sizeof *keyring);
}

static unsigned int hamm(char *map)
{
	int len = howmany(NKEYS, CHAR_BIT);
	unsigned int c = 0;
	while (len--) for (char v = map[len]; v; c++) v &= v - 1;
	return c;
}

static int load_keys(state_t *state, lc_keyring_t *keyring, char *keymap)
{
	int rc = keyring_load(state, keyring);
	unsigned int nkeys = hamm(keymap);
	if (!test_assert(rc == 0, "keyring_load() returned %i", rc)) return -1;
	if (!test_assert(keyring->nkeys == nkeys, "%zu keys loaded", keyring->nkeys)) return -1;
	return 0;
}

static int create_fakehome(char *fakehome)
{
	int rc;
	/* create fake home directory */
	if (!test_assert(mkdtemp(fakehome) != NULL, "mkdtemp()")) {
		perror("mkdtemp");
		return test_status;
	}
	rc = setenv("HOME", fakehome, 1);
	test_assert(rc == 0, "set HOME");
	return test_status;
}

static int test_keyring_add_and_del(void)
{
	char fakehome[] = "0000-0020-XXXXXX";
	char keymap[howmany(NKEYS, CHAR_BIT)];
	state_t state = {0};
	key_combo_t keys[NKEYS];
	lc_keyring_t keyring = {0};
	char *argv_00[] = { PACKAGE_NAME, "key", "add", NULL };
	FILE *f;
	int argc = sizeof argv_00 / sizeof argv_00[0] - 1;
	int rc;

	/* first, test with required argument (key) missing */
	REDIRECT_BUF
	rc = agent(&state, argc, argv_00);
	REDIRECT_OUT
	test_assert(rc == EXIT_FAILURE, "%s: agent() returned %i", __func__, rc);
	test_assert(strstr(buf, AGENT_ERR_NOKEY) != NULL, "%s() error message found in output", __func__);
	memset(&state, 0, sizeof state);

	if (create_fakehome(fakehome)) return test_status;

	/* generate some keys */
	for (int i = 0; i < NKEYS; i++) {
		rc = key_gen_combopair_hex(&keys[i]);
		test_assert(rc == 0, "generate key %i", i);
	}

	/* add keys to keyring file on disk */
	memset(keymap, 0, sizeof keymap);
	for (int i = 0; i < NKEYS; i++) {
		char *argv_01[] = { PACKAGE_NAME, "key", "add", keys[i].phex, NULL };
		argc = sizeof argv_01 / sizeof argv_01[0] - 1;
		rc = agent(&state, argc, argv_01);
		test_assert(rc == EXIT_SUCCESS, "[%i] %s: lcagent add returned %i", i, __func__, rc);
		setbit(keymap, i);
	}

	/* load keys from disk */
	rc = state_dirs(&state, fakehome);
	if (!test_assert(rc == 0, "state_dirs() returned %i", rc)) goto err_free_state;
	if (load_keys(&state, &keyring, keymap)) goto err_free_keyring;

	/* verify we have all keys */
	verify_keys(&keyring, keys, keymap);
	freakeys(&keyring);
	free_state(&state);

	/* delete some keys randomly */
	for (int i = 0; i < NKEYS; i++) {
		if (arc4random_uniform(100) < 34) continue;
		char *argv_02[] = { PACKAGE_NAME, "key", "del", keys[i].phex, NULL };
		argc = sizeof argv_02 / sizeof argv_02[0] - 1;
		rc = agent(&state, argc, argv_02);
		test_assert(rc == EXIT_SUCCESS, "[%i] %s: lcagent del returned %i", i, __func__, rc);
		clrbit(keymap, i);
	}

	/* (re)load keys from disk */
	rc = state_dirs(&state, fakehome);
	if (!test_assert(rc == 0, "state_dirs() returned %i", rc)) goto err_free_state;
	if (load_keys(&state, &keyring, keymap)) goto err_free_state;

	/* verify we have correct keys */
	verify_keys(&keyring, keys, keymap);
err_free_keyring:
	freakeys(&keyring);
err_free_state:
	free_state(&state);
	return test_status;
}

static int test_keyring_key_missing(char *action)
{
	char fakehome[] = "0000-0020-XXXXXX";
	state_t state = {0};
	FILE *f;
	char *argv_00[] = { PACKAGE_NAME, "key", action, NULL };
	int argc = sizeof argv_00 / sizeof argv_00[0] - 1;
	int rc;

	/* first, test with required argument (key) missing */
	if (create_fakehome(fakehome)) return test_status;
	REDIRECT_BUF
	rc = agent(&state, argc, argv_00);
	REDIRECT_OUT
	test_assert(rc == EXIT_FAILURE, "%s: agent() returned %i", __func__, rc);
	test_assert(strstr(buf, AGENT_ERR_NOKEY) != NULL, "%s() error message found in output", __func__);

	return test_status;
}

int main(void)
{
	char name[] = "lcagent key";

	test_name(name);

	if (test_keyring_key_missing("add")) return test_status;
	if (test_keyring_key_missing("del")) return test_status;
	if (test_keyring_add_and_del()) return test_status;

	return test_status;
}
