% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/range_speedread.R
\name{range_speedread}
\alias{range_speedread}
\title{Read Sheet as CSV}
\usage{
range_speedread(ss, sheet = NULL, range = NULL, skip = 0, ...)
}
\arguments{
\item{ss}{Something that identifies a Google Sheet:
\itemize{
\item its file id as a string or \code{\link[googledrive:drive_id]{drive_id}}
\item a URL from which we can recover the id
\item a one-row \code{\link[googledrive:dribble]{dribble}}, which is how googledrive
represents Drive files
\item an instance of \code{googlesheets4_spreadsheet}, which is what \code{\link[=gs4_get]{gs4_get()}}
returns
}

Processed through \code{\link[=as_sheets_id]{as_sheets_id()}}.}

\item{sheet}{Sheet to read, in the sense of "worksheet" or "tab". You can identify a sheet by name, with a string, or by position, with a number. Ignored if the sheet is specified via \code{range}. If neither argument specifies the sheet, defaults to the first visible sheet.}

\item{range}{A cell range to read from. If \code{NULL}, all non-empty cells are
read. Otherwise specify \code{range} as described in \href{https://developers.google.com/sheets/api/guides/concepts#a1_notation}{Sheets A1 notation}
or using the helpers documented in \link{cell-specification}. Sheets uses
fairly standard spreadsheet range notation, although a bit different from
Excel. Examples of valid ranges: \code{"Sheet1!A1:B2"}, \code{"Sheet1!A:A"},
\code{"Sheet1!1:2"}, \code{"Sheet1!A5:A"}, \code{"A1:B2"}, \code{"Sheet1"}. Interpreted
strictly, even if the range forces the inclusion of leading, trailing, or
embedded empty rows or columns. Takes precedence over \code{skip}, \code{n_max} and
\code{sheet}. Note \code{range} can be a named range, like \code{"sales_data"}, without
any cell reference.}

\item{skip}{Minimum number of rows to skip before reading anything, be it
column names or data. Leading empty rows are automatically skipped, so this
is a lower bound. Ignored if \code{range} is given.}

\item{...}{Passed along to the CSV parsing function (currently
\code{readr::read_csv()}).}
}
\value{
A \link[tibble:tibble-package]{tibble}
}
\description{
This function uses a quick-and-dirty method to read a Sheet that bypasses the
Sheets API and, instead, parses a CSV representation of the data. This can be
much faster than \code{\link[=range_read]{range_read()}} -- noticeably so for "large" spreadsheets.
There are real downsides, though, so we recommend this approach only when the
speed difference justifies it. Here are the limitations we must accept to get
faster reading:
\itemize{
\item Only formatted cell values are available, not underlying values or details
on the formats.
\item We can't target a named range as the \code{range}.
\item We have no access to the data type of a cell, i.e. we don't know that it's
logical, numeric, or datetime. That must be re-discovered based on the
CSV data (or specified by the user).
\item Auth and error handling have to be handled a bit differently internally,
which may lead to behaviour that differs from other functions in
googlesheets4.
}

Note that the Sheets API is still used to retrieve metadata on the target
Sheet, in order to support range specification. \code{range_speedread()} also
sends an auth token with the request, unless a previous call to
\code{\link[=gs4_deauth]{gs4_deauth()}} has put googlesheets4 into a de-authorized state.
}
\examples{
\dontshow{if (gs4_has_token()) withAutoprint(\{ # examplesIf}
if (require("readr")) {
  # since cell type is not available, use readr's col type specification
  range_speedread(
    gs4_example("deaths"),
    sheet = "other",
    range = "A5:F15",
    col_types = cols(
      Age = col_integer(),
      `Date of birth` = col_date("\%m/\%d/\%Y"),
      `Date of death` = col_date("\%m/\%d/\%Y")
    )
  )
}

# write a Sheet that, by default, is NOT world-readable
(ss <- sheet_write(chickwts))

# demo that range_speedread() sends a token, which is why we can read this
range_speedread(ss)

# clean up
googledrive::drive_trash(ss)
\dontshow{\}) # examplesIf}
}
